// LocalStorage anahtarı
const STORAGE_KEY = 'emergent_status_checks';

// Sayfa yükleme
document.addEventListener('DOMContentLoaded', function() {
    // İlk yükleme mesajı (console)
    console.log('Hello World');
    
    // Status listesini yükle
    loadStatusChecks();
});

// Ana sayfayı göster
function showHomePage() {
    document.getElementById('home-page').classList.add('active');
    document.getElementById('status-page').classList.remove('active');
}

// Status sayfasını göster
function showStatusPage() {
    document.getElementById('home-page').classList.remove('active');
    document.getElementById('status-page').classList.add('active');
    loadStatusChecks();
}

// UUID oluştur
function generateUUID() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });
}

// Status kontrolü oluştur
function createStatusCheck(event) {
    event.preventDefault();
    
    const clientName = document.getElementById('client-name').value.trim();
    const messageDiv = document.getElementById('form-message');
    
    if (!clientName) {
        showMessage('Client ismi gereklidir!', 'error');
        return;
    }
    
    // Yeni status kontrolü oluştur
    const statusCheck = {
        id: generateUUID(),
        client_name: clientName,
        timestamp: new Date().toISOString()
    };
    
    // Mevcut status kontrollerini al
    const statusChecks = getStatusChecks();
    
    // Yeni kontrolü ekle
    statusChecks.push(statusCheck);
    
    // LocalStorage'a kaydet
    saveStatusChecks(statusChecks);
    
    // Formu temizle
    document.getElementById('status-form').reset();
    
    // Başarı mesajı
    showMessage('Status kontrolü başarıyla oluşturuldu!', 'success');
    
    // Listeyi yenile
    loadStatusChecks();
}

// Status kontrollerini al
function getStatusChecks() {
    const data = localStorage.getItem(STORAGE_KEY);
    return data ? JSON.parse(data) : [];
}

// Status kontrollerini kaydet
function saveStatusChecks(statusChecks) {
    localStorage.setItem(STORAGE_KEY, JSON.stringify(statusChecks));
}

// Status kontrollerini yükle ve görüntüle
function loadStatusChecks() {
    const statusList = document.getElementById('status-list');
    const statusChecks = getStatusChecks();
    
    if (statusChecks.length === 0) {
        statusList.innerHTML = `
            <div class="empty-state">
                <p>Henüz status kontrolü yok</p>
                <small>Yukarıdaki formu kullanarak yeni bir status kontrolü oluşturun</small>
            </div>
        `;
        return;
    }
    
    // En son eklenenler önce gelsin
    statusChecks.reverse();
    
    // Status kontrollerini listele
    statusList.innerHTML = statusChecks.map(check => `
        <div class="status-item">
            <div class="status-item-header">
                <div>
                    <div class="client-name">${escapeHtml(check.client_name)}</div>
                    <div class="status-id">ID: ${check.id}</div>
                </div>
            </div>
            <div class="timestamp">
                📅 ${formatTimestamp(check.timestamp)}
            </div>
        </div>
    `).join('');
}

// Mesaj göster
function showMessage(message, type) {
    const messageDiv = document.getElementById('form-message');
    messageDiv.textContent = message;
    messageDiv.className = `message ${type}`;
    
    // 3 saniye sonra mesajı gizle
    setTimeout(() => {
        messageDiv.className = 'message';
    }, 3000);
}

// Timestamp formatla
function formatTimestamp(isoString) {
    const date = new Date(isoString);
    
    const options = {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        timeZone: 'Europe/Istanbul'
    };
    
    return new Intl.DateTimeFormat('tr-TR', options).format(date);
}

// XSS koruması için HTML escape
function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

// Demo veri ekle (isteğe bağlı)
function addDemoData() {
    const demoData = [
        { id: generateUUID(), client_name: 'Demo Client 1', timestamp: new Date().toISOString() },
        { id: generateUUID(), client_name: 'Demo Client 2', timestamp: new Date(Date.now() - 3600000).toISOString() },
        { id: generateUUID(), client_name: 'Demo Client 3', timestamp: new Date(Date.now() - 7200000).toISOString() }
    ];
    
    saveStatusChecks(demoData);
    loadStatusChecks();
    console.log('Demo veri eklendi!');
}